<?php
// login.php (PROD)
if (session_status() === PHP_SESSION_NONE) session_start();
$SITE_NAME = 'Passjek Affiliate';
$BASE_URL  = '.';

require __DIR__ . '/includes/db.php';
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
mysqli_set_charset($conn, 'utf8mb4');

/* ===== Helpers ===== */
function e($s){ return htmlspecialchars($s ?? '', ENT_QUOTES, 'UTF-8'); }
function table_exists(mysqli $conn, string $name): bool {
  $stmt = $conn->prepare("SELECT COUNT(*) FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
  $stmt->bind_param('s', $name);
  $stmt->execute(); $stmt->bind_result($c); $stmt->fetch(); $stmt->close();
  return $c > 0;
}
function column_exists(mysqli $conn, string $table, string $col): bool {
  $stmt = $conn->prepare("SELECT COUNT(*) FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
  $stmt->bind_param('ss', $table, $col);
  $stmt->execute(); $stmt->bind_result($c); $stmt->fetch(); $stmt->close();
  return $c > 0;
}

/* ===== Guard skema ===== */
if (!table_exists($conn, 'affiliates')) {
  http_response_code(500);
  echo '<h3 style="font-family:system-ui">Error: Tabel <code>affiliates</code> belum ada. Jalankan migrasi SQL inti terlebih dahulu.</h3>';
  exit;
}
if (!column_exists($conn, 'affiliates', 'password_hash')) {
  // Auto-migrate: tambahkan kolom password_hash jika belum ada
  $conn->query("ALTER TABLE affiliates ADD COLUMN password_hash VARCHAR(255) NULL AFTER phone");
}

/* ===== Redirect jika sudah login ===== */
if (!empty($_SESSION['affiliate']['id'])) {
  header('Location: '.$BASE_URL.'/dashboard.php'); exit;
}

/* ===== CSRF ===== */
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));

/* ===== Throttle sederhana (anti brute force) ===== */
$now = time();
if (!isset($_SESSION['login_attempts'])) $_SESSION['login_attempts'] = 0;
if (!isset($_SESSION['login_block_until'])) $_SESSION['login_block_until'] = 0;

$errors = [];
$ok_msg = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  try {
    // Blokir sementara jika melebihi batas
    if ($now < ($_SESSION['login_block_until'] ?? 0)) {
      $wait = $_SESSION['login_block_until'] - $now;
      throw new Exception('Terlalu banyak percobaan gagal. Coba lagi dalam ~'.ceil($wait/60).' menit.');
    }

    // CSRF
    if (empty($_POST['csrf']) || !hash_equals($_SESSION['csrf'], $_POST['csrf'])) {
      throw new Exception('Sesi kadaluarsa. Muat ulang halaman.');
    }

    // Input
    $email    = strtolower(trim($_POST['email'] ?? ''));
    $password = $_POST['password'] ?? '';

    // Validasi
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) $errors['email'] = 'Email tidak valid.';
    if (strlen($password) < 1) $errors['password'] = 'Password wajib diisi.';

    if (!$errors) {
      // Ambil akun
      $stmt = $conn->prepare("SELECT id, full_name, email, status, password_hash FROM affiliates WHERE email = ? LIMIT 1");
      $stmt->bind_param('s', $email);
      $stmt->execute();
      $res = $stmt->get_result();
      $user = $res->fetch_assoc();
      $stmt->close();

      if (!$user || empty($user['password_hash']) || !password_verify($password, $user['password_hash'])) {
        $_SESSION['login_attempts']++;
        if ($_SESSION['login_attempts'] >= 5) {
          $_SESSION['login_block_until'] = time() + 5*60; // 5 menit blok
          $_SESSION['login_attempts'] = 0;                // reset counter
        }
        throw new Exception('Email atau password salah.');
      }

      if ($user['status'] !== 'active') {
        throw new Exception('Akun Anda belum aktif. Hubungi admin.');
      }

      // Sukses login
      $_SESSION['affiliate'] = [
        'id'        => (int)$user['id'],
        'full_name' => $user['full_name'],
        'email'     => $user['email'],
      ];
      $_SESSION['login_attempts'] = 0;
      $_SESSION['login_block_until'] = 0;
      $ok_msg = 'Login berhasil. Mengarahkan ke dashboard...';
      header('Refresh: 1; URL='.$BASE_URL.'/dashboard.php');
    }

  } catch (mysqli_sql_exception $e) {
    $errors['fatal'] = 'Kesalahan database: '.$e->getMessage();
  } catch (Exception $e) {
    $errors['fatal'] = $e->getMessage();
  }
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Masuk — <?php echo e($SITE_NAME); ?></title>
   <link rel="icon" href="assets/img/favicon.png" />
  <meta name="description" content="Masuk ke dashboard Affiliate Passjek." />
  <link rel="icon" href="assets/img/favicon.png" />
  <meta name="theme-color" content="#0b7d62" />
  <style>
    :root{
      --green:#0b7d62; --green-ink:#075b47; --orange:#ff7a18; --orange-ink:#ff3d00;
      --ink:#111; --muted:#f5f7f6; --white:#fff; --shadow:0 14px 40px rgba(0,0,0,.12); --radius:20px;
    }
    *{box-sizing:border-box} html,body{margin:0}
    body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial,"Noto Sans",sans-serif; color:var(--ink); background:var(--muted)}
    .container{max-width:720px;margin:0 auto;padding:16px}
    .card{background:#fff; border:1px solid #e6ecea; border-radius:22px; box-shadow:var(--shadow); padding:18px; margin:16px 0}
    h1{margin:8px 0 12px; color:var(--green-ink)}
    label{font-weight:700; display:block; margin:10px 0 6px}
    .field{width:100%; padding:12px 12px; border-radius:12px; border:1px solid #dbe6e2; background:#fff; outline:none}
    .field:focus{border-color:#b8d5cd; box-shadow:0 0 0 3px rgba(11,125,98,.12)}
    .btn{display:inline-flex; align-items:center; justify-content:center; gap:10px; text-decoration:none; font-weight:800; border-radius:14px; padding:12px 16px; border:2px solid transparent; cursor:pointer}
    .btn-primary{background:linear-gradient(135deg, var(--orange), var(--orange-ink)); color:#fff}
    .btn-primary:hover{filter:brightness(.96)}
    .btn-ghost{border-color:var(--green); color:var(--green-ink); background:#fff}
    .help{font-size:12px; color:#666; margin-top:4px}
    .err{font-size:12px; color:#d71b1b; margin-top:6px; font-weight:700}
    .alert{padding:12px; border-radius:12px; margin:8px 0}
    .alert-ok{background:#ecfff5; border:1px solid #c4efd6; color:#124b33}
    .alert-err{background:#fff2f2; border:1px solid #f2cccc; color:#6b1a1a}
    .row{display:grid; grid-template-columns:1fr; gap:12px}
    .actions{display:flex; gap:10px; align-items:center; justify-content:space-between; flex-wrap:wrap}
    .toggle{margin-top:6px; font-size:12px; color:#444; display:flex; gap:8px; align-items:center}
    a.link{color:var(--green-ink); text-decoration:none}
    a.link:hover{text-decoration:underline}
  </style>
</head>
<body>
  <?php include __DIR__.'/includes/navbar.php'; ?>

  <div class="container">
    <div class="card">
      <h1>Masuk</h1>
      <p class="help">Gunakan email & password yang Anda daftarkan.</p>

      <?php if($ok_msg): ?>
        <div class="alert alert-ok"><?php echo e($ok_msg); ?></div>
      <?php endif; ?>

      <?php if(!empty($errors['fatal'])): ?>
        <div class="alert alert-err"><?php echo e($errors['fatal']); ?></div>
      <?php endif; ?>

      <form method="post" novalidate>
        <input type="hidden" name="csrf" value="<?php echo e($_SESSION['csrf']); ?>">

        <div class="row">
          <div>
            <label for="email">Email</label>
            <input id="email" name="email" class="field" type="email" value="<?php echo e($_POST['email'] ?? ''); ?>" required>
            <?php if(!empty($errors['email'])) echo '<div class="err">'.e($errors['email']).'</div>'; ?>
          </div>

          <div>
            <label for="password">Password</label>
            <input id="password" name="password" class="field" type="password" required>
            <div class="toggle">
              <input type="checkbox" id="showpwd" />
              <label for="showpwd" style="font-weight:600; margin:0">Tampilkan password</label>
            </div>
            <?php if(!empty($errors['password'])) echo '<div class="err">'.e($errors['password']).'</div>'; ?>
          </div>
        </div>

        <div class="actions" style="margin-top:12px">
          <button class="btn btn-primary" type="submit">Masuk</button>
          <div>
            <a class="link" href="register.php">Belum punya akun? Daftar</a>
            <!-- <a class="link" href="forgot.php" style="margin-left:10px">Lupa password?</a> -->
          </div>
        </div>
      </form>
    </div>
  </div>

  <?php include __DIR__.'/includes/footer.php'; ?>

  <script>
    // Tampilkan/sembunyikan password
    const showpwd = document.getElementById('showpwd');
    const pwd = document.getElementById('password');
    showpwd?.addEventListener('change', ()=>{ pwd.type = showpwd.checked ? 'text' : 'password'; });

    // Enter submit
    pwd?.addEventListener('keydown', (e)=>{ if(e.key==='Enter'){ e.target.form.submit(); }});
  </script>
</body>
</html>
